package com.surpuissant.weather2

import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.activity.viewModels
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.material3.Button
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.material3.TextField
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import com.squareup.moshi.Moshi
import com.squareup.moshi.kotlin.reflect.KotlinJsonAdapterFactory
import com.surpuissant.weather2.ui.theme.Weather2Theme
import okhttp3.OkHttpClient

class MainActivity : ComponentActivity() {

    private val okHttpClient by lazy { OkHttpClient() }

    object MoshiProvider {
        val moshi: Moshi = Moshi.Builder().add(KotlinJsonAdapterFactory()).build()
    }

    private val viewModel: WeatherViewModel by viewModels {
        RawWeatherViewModelFactory(okHttpClient)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()


        setContent {
            Weather2Theme {
                Scaffold(modifier = Modifier.fillMaxSize()) { innerPadding ->
                    Row {
                        WeatherScreen(
                            modifier = Modifier.padding(innerPadding),
                            viewModel = viewModel,
                            city= ""
                        )
                    }
                }
            }
        }
    }
}

@Composable
fun WeatherScreen(modifier: Modifier, city: String, viewModel: WeatherViewModel) {

    val preview by viewModel.preview.collectAsStateWithLifecycle()
    var myCity by remember { mutableStateOf(city) }
    var isLoading by remember { mutableStateOf(false) }

    Surface(modifier = modifier.fillMaxSize().padding(12.dp)) {

        Column {
            EditCity(
                value = myCity,
                onValueChange = { newCity ->
                    myCity = newCity.trim()
                },
                onDone = {
                    if(!myCity.isBlank()) {
                        isLoading = true
                        viewModel.refresh(myCity) {
                            isLoading = false
                        }
                    }
                })

            Spacer(modifier = Modifier.height(24.dp))

            preview?.let { unwrappedPreview ->
                Column {
                    Text(text = unwrappedPreview.name)
                    Text(text = unwrappedPreview.main.temp.toString())
                    Text(text = unwrappedPreview.weather.first().description)
                }
            } ?: run {
                if(isLoading) {
                    Row(
                        horizontalArrangement = Arrangement.Center
                    ) {
                        CircularProgressIndicator(modifier = Modifier.size(18.dp))
                        Spacer(Modifier.width(12.dp))
                        Text(text = "Chargement en cours…")
                    }
                } else {
                        Text(text = "Veuillez entrer une ville pour commencer")
                }
            }
        }
    }
}






@Composable
fun EditCity(value: String, onValueChange: (String) -> Unit, onDone: () -> Unit) {

    Row {
        TextField(
            modifier = Modifier.width(220.dp),
            value = value,
            onValueChange = onValueChange,
            label = { Text("Entrer une ville") },
            singleLine = true,
            keyboardOptions = KeyboardOptions(imeAction = ImeAction.Done),
            keyboardActions = KeyboardActions(onDone = { onDone() })
        )

        Spacer(modifier = Modifier.width(12.dp))

        Button(onClick = onDone) {
            Text("Rechercher")
        }
    }

}